package jcircus.environment;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;

import jcircus.exceptions.NoNameTypeAnnotationException;
import jcircus.util.CircusType;
import jcircus.util.NameType;
import jcircus.util.Util;
import net.sourceforge.czt.z.ast.DeclName;
import net.sourceforge.czt.z.ast.Name;
import net.sourceforge.czt.z.util.Factory;

/**
 * NameTypeEnv.java
 *
 * @author Angela Freitas
 */
public class NameTypeEnv {
    
    /**
     * DeclName -> CircusType
     */
    private Hashtable hashtable_;
    
    private Factory factory_;
    
    /**
     * Constructor
     *
     */
    public NameTypeEnv() {
        this.hashtable_ = new Hashtable();
        factory_ = new Factory();
    }
    
    /**
     * Insert a name and type in the environment.
     *
     * @param nameEnv
     * @param circusType
     *
     * @throws Exception	If the name does hot have an NameType annotation.
     */
    public void put(DeclName declName, CircusType circusType) throws NoNameTypeAnnotationException {
        
        if (declName.getAnn(NameType.class) == null) {
            throw new NoNameTypeAnnotationException(declName.toString());
        }
        
        this.hashtable_.put(declName, circusType);
    }
    
    /**
     *
     * @param string
     * @return
     */
    public CircusType getCircusType(String string) {
        
        CircusType circusType = null;
        
        DeclName n = this.factory_.createDeclName(string); //new DeclName(string);
        DeclName t;
        
        Enumeration keys = this.hashtable_.keys();
        
        while (keys.hasMoreElements()) {
            
            t = (DeclName) keys.nextElement();
            
            if (t.equals(n)) {
                
                circusType = (CircusType) this.hashtable_.get(t);
                break;
            }
        }
        return circusType;
    }
    
    /**
     * Returns the NameType associated with the given declName.
     * In case the declName is not defined in this environment, returns null.
     * 
     * @param string
     * @return
     */
    public NameType getNameType(String string) {
        
        NameType nameType = null;
        
        DeclName t;
        
        Enumeration keys = this.hashtable_.keys();
        
        while (keys.hasMoreElements()) {
            
            t = (DeclName) keys.nextElement();
            
            if (t.toString().equals(string)) {
                
                nameType = (NameType) t.getAnn(NameType.class);
                break;
            }
        }
        
        return nameType;
    }
    
    /**
     *
     * @param channelName
     */
    public void remove(String channelName) {
        this.hashtable_.remove(channelName);
    }
    
    /**
     *
     * @param newChannels
     * @param oldChannels
     * @return
     */
    public NameTypeEnv substitute(List newChannels, List oldChannels) {
        
        NameTypeEnv r = new NameTypeEnv();
        Name oldChannelName, newChannelName;
        DeclName declName;
        int index;
        CircusType circusType;
        NameType nameType;
        
        r.putAll(this);
        
        for (int i = 0; i < oldChannels.size(); i++) {
            
            oldChannelName = (Name) oldChannels.get(i);
            newChannelName = (Name) newChannels.get(i);
            
            if (r.containsKey(oldChannelName.toString())) { 
                nameType = r.getNameType(oldChannelName.toString());
                circusType = r.getCircusType(oldChannelName.toString());
                
                r.remove(oldChannelName.toString());
                
                declName = this.factory_.createDeclName(newChannelName.toString());
                Util.addNameTypeAnn(declName, nameType);
                
                r.put(declName, circusType);
            }
        }
        
        return r;
    }
    
    /**
     *
     */
    public void putAll(NameTypeEnv typeEnvironment) {
        
        Iterator iterator = typeEnvironment.iteratorKeys();
        DeclName declName;
        CircusType circusType;
        
        while(iterator.hasNext()) {
            declName = (DeclName) iterator.next();
            circusType = typeEnvironment.getCircusType(declName.toString());
            
            this.put(declName, circusType);
        }
    }
    
    /**
     *
     * @param declName
     * @return
     */
    public boolean containsKey(String declName) {
        
        boolean r = false;
        
        DeclName t;
        Enumeration keys = this.hashtable_.keys();
        
        while (keys.hasMoreElements()) {
            
            t = (DeclName) keys.nextElement();
            
            if (t.toString().equals(declName)) {
                r = true;
            }
        }
        return r;
    }
    
    /**
     *
     * @return
     */
    public Enumeration keys() {
        
        return this.hashtable_.keys();
    }
   
    /**
     *
     */
    public Iterator iteratorKeys() {
        
        return this.hashtable_.keySet().iterator();
    }
    
    /**
     *
     */
    public void print() {
        
        DeclName declName;
        CircusType type;
        Enumeration en = this.hashtable_.keys();
        
        System.out.println("-------------");
        while (en.hasMoreElements()) {
            
            declName = (DeclName) en.nextElement();
            type = (CircusType) this.hashtable_.get(declName);
            
            System.out.println("DeclName: " + declName + ", CircusType: " + type);
        }
        System.out.println("-------------");
    }
    
}